#!/bin/bash

# If the host is running systemd-resolved (as is the default on Ubuntu 18.04,
# for instance) then we need to use a different resolv.conf. See issue:
#	https://github.com/kubernetes/kubeadm/issues/273
# TODO (nsteph): --resolv-conf is deprecated in favor of the kubelet config
RESOLV_CONF_FLAG=""
SYSTEMD_RESOLV_CONF="/run/systemd/resolve/resolv.conf"

# Reset PATH to exclude binaries under the container rootfs directory.
export PATH="/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin"

POD_INFRA_CONTAINER_IMAGE=$1

cat <<EOF >/tmp/cri-dockerd-mke.service
[Unit]
Description=MKE CRI Interface for Docker Application Container Engine
Documentation=https://docs.mirantis.com
After=network-online.target firewalld.service docker.service
Wants=network-online.target
Requires=cri-dockerd-mke.socket

[Service]
Type=notify
ExecStart=/usr/local/bin/cri-dockerd-mke --log-level ${LOG_LEVEL} --container-runtime-endpoint fd:// --pod-infra-container-image=$POD_INFRA_CONTAINER_IMAGE
ExecReload=/bin/kill -s HUP \$MAINPID
TimeoutSec=0
RestartSec=2
Restart=always

# Note that StartLimit* options were moved from "Service" to "Unit" in systemd 229.
# Both the old, and new location are accepted by systemd 229 and up, so using the old location
# to make them work for either version of systemd.
StartLimitBurst=3

# Note that StartLimitInterval was renamed to StartLimitIntervalSec in systemd 230.
# Both the old, and new name are accepted by systemd 230 and up, so using the old name to make
# this option work for either version of systemd.
StartLimitInterval=60s

# Having non-zero Limit*s causes performance problems due to accounting overhead
# in the kernel. We recommend using cgroups to do container-local accounting.
LimitNOFILE=infinity
LimitNPROC=infinity
LimitCORE=infinity

# Comment TasksMax if your systemd version does not support it.
# Only systemd 226 and above support this option.
TasksMax=infinity
Delegate=yes
KillMode=process

[Install]
WantedBy=multi-user.target
EOF

cat <<EOF >/tmp/cri-dockerd-mke.socket
[Unit]
Description=MKE CRI Docker Socket for the API
PartOf=cri-dockerd-mke.service

[Socket]
ListenStream=%t/cri-dockerd-mke.sock
SocketMode=0660
SocketUser=root
SocketGroup=docker

[Install]
WantedBy=sockets.target
EOF

restart=false

# Copy the file if it doesn't exist. Update the file if it exists and it's different.
if [[ ! -f /usr/local/bin/cri-dockerd-mke ]]; then
	cp ./cri-dockerd-mke /usr/local/bin/
	restart=true
else
	cmp --silent /usr/local/bin/cri-dockerd-mke ./cri-dockerd-mke
	if [[ $? -ne 0 ]]; then
		systemctl stop cri-dockerd-mke.service && cp ./cri-dockerd-mke /usr/local/bin/
		restart=true
	fi
fi

[[ -f /usr/lib/systemd/system/cri-dockerd-mke.socket ]] || cp /tmp/cri-dockerd-mke.socket /usr/lib/systemd/system

if [[ ! -f /usr/lib/systemd/system/cri-dockerd-mke.service ]]; then
	cp /tmp/cri-dockerd-mke.service /usr/lib/systemd/system
	restart=true
else
	cmp --silent /tmp/cri-dockerd-mke.service /usr/lib/systemd/system/cri-dockerd-mke.service
	if [[ $? -ne 0 ]]; then
 		cp /tmp/cri-dockerd-mke.service /usr/lib/systemd/system
		restart=true
	fi
fi

# Enable or restart cri-dockerd
if [[ -z "$(ps -eoargs | grep '[c]ri-dockerd-mke' | grep -v process-proxy | grep -v kubelet)" || $restart = "true" ]]; then
	systemctl daemon-reload
	if [ -z "$(pgrep cri-dockerd-mke)" ]; then
		systemctl enable cri-dockerd-mke.socket --now
		systemctl enable cri-dockerd-mke.service --now
	else
		systemctl restart cri-dockerd-mke.socket
		systemctl restart cri-dockerd-mke.service
	fi
fi

# Note that the kubelet runs in host PID mode, so we can check if a process
# named systemd-resolved is running on the host machine.
if [ -n "$(pgrep systemd-resolve)" ]; then
	if [ -f "/run/systemd/resolve/resolv.conf" ]; then
		echo "systemd-resolved is running, so using --resolv-conf=${SYSTEMD_RESOLV_CONF} for kubelet."
		RESOLV_CONF_FLAG="--resolv-conf=${SYSTEMD_RESOLV_CONF}"
	else
		echo "systemd-resolved is running, but ${SYSTEMD_RESOLV_CONF} file is not present, so not setting --resolv-conf on the kubelet."
	fi
else
	echo "Using default resolv-conf location."
fi

# The Kubelet connects to the Kube API server on a manager node using the
# reverse proxy running in the local ucp-proxy container. It connects to
# that proxy using the domain name 'proxy.local'. For most Kube component
# containers, we add this as an "ExtraHosts" entry in the container's host
# config, but Kubelet is running as a host process and uses the host's
# /etc/hosts file. Instead, we append an entry to /etc/hosts.
if grep -q "proxy.local" /etc/hosts; then
	sed -i "/proxy.local/ s/.*/${PROXY_DOT_LOCAL_IP}\tproxy.local/g" /etc/hosts
else
	echo -e "\n# BEGIN Docker UCP settings" >>/etc/hosts
	echo "${PROXY_DOT_LOCAL_IP} proxy.local" >>/etc/hosts
	echo "# END Docker UCP settings" >>/etc/hosts
fi

# Kubelet binds an ephemeral port to localhost for proxying streaming
# operations (e.g. kubectl exec) to the container runtime.  If this port
# overlaps with an active NodePort, streaming operations won't work.  Address
# this by reserving the NodePort range on the host.  The kernel API
# automatically performs a union between the specified ip_local_reserved_ports
# and any existing config, so it's harmless to invoke this every time kubelet
# starts.
if [ -z "$(sysctl --values net.ipv4.ip_local_reserved_ports)" ]; then
	sysctl -w "net.ipv4.ip_local_reserved_ports=${NODEPORT_RANGE}"
else
	sysctl -w "net.ipv4.ip_local_reserved_ports=$(sysctl --values net.ipv4.ip_local_reserved_ports),${NODEPORT_RANGE}"
fi

modprobe nf_conntrack_proto_gre

DISTRO=$(grep -Po "^NAME=\"?\K[a-zA-Z\s]+" /etc/os-release)
echo "Checking for conntrack(-tools)"
if ! command -v conntrack >/dev/null; then
	# Get the name of the distro
	# Do the right thing to install conntrack tools
	if [[ "${DISTRO}" == "Ubuntu" ]]; then
		export DEBIAN_NONINTERACTIVE=true
		echo "Updating package index"
		apt-get update >/dev/null 2>&1
		echo "Installing conntrack"
		apt-get install -y conntrack 2>&1
	else
		echo "Installing conntrack-tools"
		yum -y install conntrack-tools >/dev/null 2>&1
	fi
fi

# Try to load the conntrack module to track protocol 47/GRE
modprobe nf_conntrack_proto_gre
echo "Starting Kubelet..."
# Pass all but the first argument (the pause image, which is used for
# invoking cri-dockerd above but not needed by the kubelet itself)
exec ./kubelet "${@:2}" $RESOLV_CONF_FLAG
