#!/usr/bin/env python3

import argparse
from datetime import datetime
import glob
import json
import pprint
import os
import re
import shutil
import subprocess
import sys
import tarfile
import time
import traceback
import yaml

tempdir_default = "/dsinfo"
output_default = "{}/dsinfo.txt".format(tempdir_default)
json_default = "{}/dsinfo.json".format(tempdir_default)
default_handle = None
jsondata = {}

networks = 0
net_containers = 0
ip_overlap = 0
nsdir = "/var/run/docker/netns"


def decode(string):
    try:
        string = string.decode("utf-8")
    except (UnicodeDecodeError, AttributeError):
        pass
    return string


def log(output="", handle=None):
    if handle is None:
        handle = default_handle
    # Force cast this to a string since subprocess on python3 returns bytes
    output = decode(output)
    print(output, file=handle)


def log_header(info):
    log("=" * 25)
    log("")
    log(info)
    log("=" * 25)
    log("")


def log_stderr(output):
    sys.stderr.write(
        "{} {}\n".format(datetime.now().strftime("%H:%M:%S.%f"), decode(output))
    )
    sys.stderr.flush()


def validate_date(str):
    try:
        return datetime.strptime(str, "%Y-%m-%d %H:%M:%S")
    except ValueError:
        raise argparse.ArgumentTypeError("Invalid date format: {0!r}".format(str))


class DSinfo(object):
    batch = False
    docker = False
    audit = False
    ssd = False
    handle = None

    loglines, tempdir, tarfile = None, None, None

    def __init__(self):
        self.parse_args()
        self.makedirs()

        if not self.batch:
            self.print_leader()
            self.check_debug()

    @staticmethod
    def exec_cmd(cmd, json_category=None, shell=False):
        log("")
        if type(cmd) is list:
            log_stderr("    {}".format(" ".join(cmd)))
        else:
            log_stderr("    {}".format(cmd))

        if (type(cmd) is list and shutil.which(cmd[0])) or shell:
            try:
                output = decode(subprocess.check_output(cmd, shell=shell))
                log(output)
            except subprocess.CalledProcessError as e:
                log("An exception was encountered running this command: {}".format(cmd))
                log("OUTPUT: {}, STDERR: {}".format(e.output, e.stderr))
                return []

            if cmd[0] == "docker" and cmd[1] != "exec":
                json_cmd = cmd + ["--format", "{{json .}}"]
                json_output = decode(subprocess.check_output(json_cmd))
                try:
                    json_output = json.loads(json_output)
                except json.decoder.JSONDecodeError:
                    # Some docker commands return a JSON stream, which is silly, but
                    # we can deal
                    json_output = [json.loads(line) for line in json_output.splitlines()]
                if json_category:
                    jsondata[json_category] = json_output
            else:
                if json_category:
                    jsondata[json_category] = output.splitlines()
                return output.splitlines()
            return json_output
        else:
            log("'{}' command not found".format(cmd[0]))

    @staticmethod
    def print_leader():
        print()
        print("*************************** dsinfo.py ***************************")
        print()
        print("Beginning data collection ...")
        print()

    def check_debug(self):
        try:
            output = decode(subprocess.check_output(["docker", "info", "--format", "{{json .}}"]))
        except subprocess.CalledProcessError as e:
            log_stderr("Couldn't get docker info: {}".format(e.output))
            return
        if not json.loads(output)["Debug"]:
            print()
            print("The Docker daemon is not running with debug logging enabled!")
            print()
            string = "Enable debug logging and restart the docker daemon? [y/n]: "
            val = input(string)
            val = val.lower()

            if val == "y" or val == "yes":
                self.set_docker_debug()
            else:
                print()
                print("Continuing without enabling debug logging ...")

    @property
    def get_osrelease(self):
        info = {}
        with open("/etc/os-release", "r") as f:
            for line in f:
                k, v = line.rstrip().split("=")
                # Strip out the unnecessary quotes some distros use around field values
                if v.startswith('"'):
                    v = v[1:-1]
                    info[k] = v
        return info

    def set_docker_debug(self):
        osname = self.get_osrelease["NAME"]
        if (
                osname == "CentOS Linux"
                or osname == "Fedora"
                or osname == "Red Hat Enterprise Linux Server"
        ):
            optname = "OPTIONS"
            optpath = "/etc/sysconfig/docker"
            cmd = ["systemctl", "restart", "docker"]
        elif osname == "Debian GNU/Linux" or osname == "Ubuntu":
            optname = "DOCKER_OPTS"
            optpath = "/etc/default/docker"
            cmd = ["service", "docker", "restart"]
        else:
            print()
            print("Can't determine the location of the Docker config file")
            print("Enable debug logging manually and restart the Docker daemon")
            sys.exit(1)
        with open(optpath, "r") as f:
            lines = f.readlines()
        if not [x for x in lines if re.match(r"^{}".format(optname), x)]:
            lines += "{}='-D".format(optname)
        else:
            for i in range(len(lines)):
                if re.match(r"^{}".format(optname), lines[i]):
                    # Dunno if it'll be ' or ", so capture both and be smart"
                    lines[i] = re.sub(
                        r'{}=([\'"])(.*?)[\'"]'.format(optpath), r"{}=\1-D \2\1", lines[i]
                    )
        with open(optpath, "w") as f:
            [f.write("{}\n".format(line)) for line in lines]

        try:
            subprocess.check_call(cmd)
            print()
            print("Debug logging enabled and daemon restarted")
            print("Run the 'dsinfo.py' data collection script again when you're ready...")
        except subprocess.CalledProcessError:
            print("Could not restart the Docker daemon!")
            sys.exit(1)

    @property
    def now(self):
        # Approximate the `date` command as closely as we can to keep
        # compatibility with existing support logs
        return datetime.now().strftime("%a %d %b %Y %H:%M:%S %p %z")

    def run(self):
        try:
            self.shim_logs()
        except BaseException:
            log_stderr("Could not shim collect logs")

        try:
            # Remove the old stacktrace files
            subprocess.call(["docker", "exec", "ucp-controller", "sh", "-c", "rm -f /tmp/diag/*"])
            # Generate new stacktrace of ucp-controller
            subprocess.call(
                ["docker", "kill", "-s", "USR1", "ucp-controller"], stdout=open(os.devnull, "w")
            )
        except subprocess.CalledProcessError as e:
            log_stderr("Couldn't SIGUSR1 the UCP controller: {}".format(e.output))

        funcs = {
            "System Information": self.system_information,
            "Cluster Info": self.cluster_info,
            "Docker Logs": self.get_docker_logs,
            "Docker Events": self.get_docker_events,
            "System Logs": self.get_system_logs,
            "CRI-dockerd Logs": self.get_cri_dockerd_logs,
            "Containerd Logs": self.get_containerd_logs,
        }

        for k, v in funcs.items():
            try:
                funcs[k]()
            except BaseException:
                log_stderr("Could not check {}".format(v))

        if self.stacktraces:
            try:
                if os.path.isfile("/var/run/docker.pid"):
                    self.get_stacktraces()
            except BaseException:
                log_stderr("Could not get stack traces: {}".format(traceback.format_exc()))

        try:
            if self.docker:
                self.get_container_info()
        except BaseException as e:
            log_stderr("Could not get container info: {}".format(traceback.format_exc()))

        try:
            output = None

            if os.path.isfile("/etc/dci_deployment.info"):
                log_stderr("Collecting Docker Certified Infrastructure Info")
                log_header("Docker Certified Infrastructure")
                output = self.exec_cmd(["cat", "/etc/dci_deployment.info"])
            jsondata["dci_deployment"] = output if output else "Not present"
        except BaseException:
            log_stderr("Could not get DCI info")

        funcs = {
            "DTR": self.check_dtr,
            "System Metrics": self.system_metrics,
            "Networking Info": self.networking_info,
        }

        for k, v in funcs.items():
            try:
                funcs[k]()
            except BaseException:
                log_stderr("Could not check {}".format(v))

        if self.ssd:
            log_stderr(
                "Performing SSD control-plane and datapath consistency check on a node ####"
            )
            log("\n#### SSD control-plane and datapath consistency check on a node ####")
            try:
                netnames = decode(
                    subprocess.check_output(
                        [
                            "docker",
                            "network",
                            "ls",
                            "-f",
                            "driver=overlay",
                            "--format",
                            "{{.Name}}",
                        ]
                    ).splitlines()
                )
                for net in netnames:
                    log("## {} ##".format(net))
                    lines = decode(subprocess.check_output(["ssd", net]))
                    log(lines)
                    jsondata["ssd"] = lines.splitlines()
                    log()
            except subprocess.CalledProcessError as e:
                log_stderr("Could not check Docker networks: {}".format(e.output))
        else:
            jsondata["ssd"] = []

        log("\n\n==SUMMARY==")
        log("\t Processed {} networks".format(networks))
        log("\t IP overlap found: {}".format(ip_overlap))
        log("\t Processed {} running containers".format(net_containers))

        if "CNI_PROVIDER" in os.environ and os.environ["CNI_PROVIDER"] == "calico":
            try:
                self.process_calico()
            except BaseException as e:
                # Calico can also be flaky in CI -- skip this if it fails so we get the rest
                log_stderr("Could not gather Calico info: {}".format(e.output))
                pass

        try:
            log_stderr("Collecting Docker Daemon Certificates")
            for cert in glob.glob("/etc/docker/certs.d/*"):
                try:
                    rename = re.sub(r"/", "_", cert[2:])
                    os.symlink(cert, "/{}".format(rename))
                except BaseException:
                    log_stderr("Could not symlink {}".format(cert))
        except BaseException:
            log_stderr("Could not list Docker Daemon certificates")

        log_stderr("Collecting UCP Component Certificates")
        dump = "{}/certdump.txt".format(self.tempdir)
        f = open(dump, "w")
        try:
            output = decode(subprocess.check_output(["certdump", "--scan"]))
            f.write(output)
        except BaseException as e:
            f.write("`certdump --scan` failed: {}".format(traceback.format_exc()))
        finally:
            f.close()

        log_stderr("Collecting Current UCP Configuration")
        try:
            output = decode(subprocess.check_output(["docker", "info", "-f", "{{json .}}"]))
            docker_root = json.loads(output)["DockerRootDir"]
        except subprocess.CalledProcessError as e:
            log_stderr("Coudln't check docker info: {}".format(e.output))
            docker_root = "/var/lib/docker"

        node_certs_dir = "{}/volumes/ucp-node-certs/_data".format(docker_root)
        files = ["node-reconciler-config", "cluster-reconciler-config"]
        for f in files:
            try:
                fname = "{}/{}.json".format(node_certs_dir, f)
                if os.path.exists(fname) and os.stat(fname).st_size == 0:
                    shutil.copy(fname, "{}/{}.json".format(self.tempdir, f))
                    jsondata[f] = json.loads(open("{}/{}.json".format(node_certs_dir, f)).read())
                else:
                    with open("{}/{}.json".format(self.tempdir, f), "w") as g:
                        g.write("{} was empty".format(fname))
            except BaseException as e:
                with open("{}/{}.json".format(self.tempdir, f), "w") as g:
                    g.write("Failed to copy file: {}".format(traceback.format_exc()))

        try:
            self.get_xfs_info()
        except BaseException as e:
            log_stderr("Failed to gather xfs_info")
            log_stderr(str(e))
            traceback.print_exc()

        self.finish()

    def finish(self):
        with open("{}/dsinfo.json".format(self.tempdir), "w") as f:
            json.dump(jsondata, f, indent=4)

        log_stderr("Archiving Collected Support Diagnostics Info")
        tarball = tarfile.open(self.tarfile, "w|gz")
        tarball.add(self.tempdir, arcname=os.path.basename(self.tempdir), recursive=True)
        tarball.close()

        if self.batch:
            sys.stdout.buffer.write(open(self.tarfile, "rb").read())
        else:
            print(
                """Data collection complete ...

Notes
=====

This script created the directories:
  - /dsinfo
  - /dsinfo/dtr
  - /dsinfo/dtr/generatedConfigs
  - /dsinfo/dtr/logs
  - /dsinfo/inspect
  - /dsinfo/logs
  - /dsinfo/shimlogs

This script collected the following:
  - Docker daemon configuration and logs
  - Inspect results and logs from the last 20 containers
  - Miscellaneous system information (Output to: /dsinfo/report.md)

All files/directories were compressed to: /dsinfo.tar.gz

---------------------------------------------------------------------------------

*** Important ***

Before sharing the dsinfo.tar.gz archive, review all collected files for
private information and edit/delete if necessary.

If you do edit/remove any files, recreate the tar file with the following command:

  sudo tar -zcf /dsinfo.tar.gz /dsinfo"""
            )

    def process_calico(self):
        log_stderr("Collecting Calico CNI Networking Info")

        if not shutil.which("calicoctl"):
            log_stderr("`calioctl` was not found. Aborting...")
            log("`calioctl` was not found. Aborting...")
            return
        env = os.environ.copy()
        env["ETCD_ENDPOINTS"] = "proxy.local:12378"
        env["ETCD_KEY_FILE"] = "/ucp-kv-certs/key.pem"
        env["ETCD_CA_CERT_FILE"] = "/ucp-kv-certs/ca.pem"
        env["ETCD_CERT_FILE"] = "/ucp-kv-certs/cert.pem"

        # Process both 'mirantis' and 'docker' (in that order) so we catch both versions
        for org in ["mirantis", "docker"]:
            image = org + "/ucp-calico-node"
            tag = decode(
                subprocess.check_output(["docker", "image", "ls", image, "--format", "{{.Tag}}"])
            )
            if tag:
                # Only use the most recently built
                current_tag = tag.splitlines()[0]
                subprocess.call(
                    ["docker", "image", "tag", "{}:{}".format(image, current_tag), "calico/node"]
                )
                break
        if not tag:
            log_stderr("warning: could not find calico node image. calico diags may be incomplete.")
        subprocess.call(
            ["calicoctl", "node", "diags", "--allow-version-mismatch"],
            env=env,
            stdout=subprocess.DEVNULL if self.batch else None,
            stderr=subprocess.DEVNULL if self.batch else None,
        )
        os.makedirs("{}/cni/calico".format(self.tempdir))

        for file in glob.glob("/tmp/calico*/diagnostics/*"):
            if os.path.isfile(file):
                shutil.copy(
                    file, "{}/cni/calico/{}.txt".format(self.tempdir, os.path.basename(file))
                )
            else:
                shutil.copytree(
                    file, "{}/cni/calico/{}".format(self.tempdir, os.path.basename(file))
                )

        # Trim calico logs to the log line length limit
        logfilter = "ls {}/cni/calico/logs/cni/*.log | xargs sed -i -ne':a;$p;N;{},$D;ba'".format(self.tempdir,
                                                                                                  int(self.loglines) + 1)
        subprocess.call(logfilter, shell=True)

        jsondata["calico"] = {}
        output = decode(subprocess.check_output(["calicoctl", "node", "status", "--allow-version-mismatch"], env=env))
        with open("{}/cni/calico/node_status.txt".format(self.tempdir), "w") as f:
            f.write(output)
        jsondata["calico"]["node_status"] = output.splitlines()

        cmd = ["calicoctl", "get", "workloadEndpoint", "-a", "--allow-version-mismatch"]
        with open("{}/cni/calico/workload_endpoint.txt".format(self.tempdir), "w") as f:
            f.write(decode(subprocess.check_output(cmd + ["-o", "wide"], env=env)))
        jsondata["calico"]["workload_endpoint"] = decode(
            subprocess.check_output(cmd + ["-o", "json"], env=env)
        )

        cmd = ["calicoctl", "get", "bgpPeer", "--allow-version-mismatch"]
        with open("{}/cni/calico/bgp_peers.txt".format(self.tempdir), "w") as f:
            f.write(decode(subprocess.check_output(cmd, env=env)))
        jsondata["calico"]["bgp_peers"] = decode(
            subprocess.check_output(cmd + ["-o", "json"], env=env)
        )

        if env["CLOUD_PROVIDER"] and env["CLOUD_PROVIDER"] == "azure":
            os.makedirs("{}/cni/azure".format(self.tempdir))
            subprocess.call(
                [
                    "docker",
                    "cp",
                    "ucp-kubelet:/var/log/azure-vnet-ipam.log",
                    "{}/cni/azure/".format(self.tempdir),
                ]
            )

        # Set perms for the support bundle so they're all readable
        for root, dirs, files in os.walk("{}/cni".format(self.tempdir)):
            for directory in dirs:
                os.chmod(os.path.join(root, directory), 755)
            for file in files:
                os.chmod(os.path.join(root, file), 755)

    def networking_info(self):
        log_stderr("Collecting Networking Info")
        log()
        log_header("Network Information")
        log()

        jsondata["networking"] = {}

        utils = ["docker", "nsenter", "bridge", "iptables", "ipvsadm", "ip", "ssd", "jq"]
        for u in utils:
            if not shutil.which(u):
                log("This tool requires {}".format(u))

        net_verbose = (
            "--verbose"
            if "--verbose"
               in decode(subprocess.check_output(["docker", "network", "inspect", "--help"]))
            else ""
        )
        self.host_networking()
        self.overlay_networking(net_verbose)
        self.bridge_networking(net_verbose)
        self.container_network_configuration()

    def container_network_configuration(self):
        global net_containers

        jsondata["containers"] = {}

        log_stderr("Collecting Container Network Configuration")
        log("Container network configuration")

        containers = decode(
            subprocess.check_output(
                ["docker", "container", "ls", "-a", "--format", "'{{.ID}} {{.Status}}'"]
            )
        ).splitlines()
        containers = [c.strip("'") for c in containers]
        for c in containers:
            fields = c.split()
            container_id, status = fields[0], fields[1]
            jsondata["containers"][container_id] = {}
            jsondata["containers"][container_id]["status"] = status

            if not status == "Up":
                continue

            log_stderr("Inspecting Container {}".format(container_id))
            log("ccc Container {} state: {}".format(container_id, status))
            # This is utterly insane and I'd love to kill it, but we'll keep it for consistency
            try:
                log(
                    decode(
                        subprocess.check_output(
                            "docker container inspect %s --format "
                            '\'Name:{{json .Name | printf "%%s\\n"}}Id:{{json .Id | '
                            'printf "%%s\\n"}}Hostname:{{json .Config.Hostname | '
                            'printf "%%s\\n"}}CreatedAt:{{json .Created | printf "%%s\\n"}}'
                            'State:{{json .State|printf "%%s\\n"}}'
                            "RestartCount:{{json .RestartCount | "
                            'printf "%%s\\n" }}Labels:{{json .Config.Labels | printf "%%s\\n"}}'
                            "NetworkSettings:{{json .NetworkSettings}}' | "
                            'sed \'/^State:/ {{s/\\"/QUOTE/g; s/,"Output":"[^"]*"//g;}}\''
                            % container_id,
                            shell=True,
                        )
                    )
                )
                data = decode(
                    subprocess.check_output(
                        ["docker", "container", "inspect", container_id, "--format", "{{json .}}"]
                    )
                )
                jsondata["containers"][container_id]["inspect"] = json.loads(data)
                self.get_container_network_info(
                    container_id,
                    jsondata["containers"][container_id]["inspect"]["NetworkSettings"][
                        "SandboxKey"
                    ],
                )
                net_containers += 1
            except subprocess.CalledProcessError as e:
                # More timing
                if "No such container" in decode(e.output):
                    pass
                if e.stderr and "No such container" in decode(e.stderr):
                    pass
                else:
                    log_stderr(decode(e.stderr))

    def get_container_network_info(self, container_id, container_net):
        if not container_net:
            return
        tables = ["nat", "mangle"]
        jsondata["containers"][container_id]["iptables"] = {}
        jsondata["containers"][container_id]["ip"] = {}
        jsondata["containers"][container_id]["bridges"] = []

        jsondata["containers"][container_id]["ip"]["ipv4"] = self.exec_cmd(
            ["nsenter", "--net={}".format(container_net), "ip", "-o", "-4", "address", "show"]
        )
        jsondata["containers"][container_id]["ip"]["ipv4_route"] = self.exec_cmd(
            ["nsenter", "--net={}".format(container_net), "ip", "-4", "route", "show"]
        )
        jsondata["containers"][container_id]["ip"]["ipv4_neigh"] = self.exec_cmd(
            ["nsenter", "--net={}".format(container_net), "ip", "-4", "neigh", "show"]
        )

        for t in tables:
            cmd = "nsenter --net={} iptables -w1 -n -v -L -t {} | grep -v '^$'".format(
                container_net, t
            )
            jsondata["containers"][container_id]["iptables"][t] = self.parse_iptables(
                self.exec_cmd(cmd, shell=True)
            )

        if shutil.which("ipvsadm"):
            jsondata["containers"][container_id]["ipvsadm"] = self.exec_cmd(
                ["nsenter", "--net={}".format(container_net), "ipvsadm", "-l", "-n"]
            )
        else:
            jsondata["containers"][container_id]["ipvsadm"] = []

    def host_networking(self):
        tables = ["filter", "nat", "mangle"]

        jsondata["networking"]["system"] = {}
        jsondata["networking"]["system"]["iptables"] = {}

        log_stderr("Collecting Host iptables State")
        log("Host iptables")
        for t in tables:
            cmd = "iptables -w1 -n -v -L -t {} | grep -v '^$'".format(t)
            log(cmd)
            jsondata["networking"]["system"]["iptables"][t] = self.parse_iptables(
                self.exec_cmd(cmd, shell=True)
            )
        log()

        log_stderr("Collecting Host links Addresses and Routes")
        log("Host links addresses and routes")

        jsondata["networking"]["system"]["ip"] = {}

        jsondata["networking"]["system"]["ip"]["link"] = self.exec_cmd(
            ["ip", "-o", "link", "show"]
        )
        jsondata["networking"]["system"]["ip"]["ipv4"] = self.exec_cmd(
            ["ip", "-o", "-4", "address", "show"]
        )
        jsondata["networking"]["system"]["ip"]["ipv4_route"] = self.exec_cmd(
            ["ip", "-4", "route", "show"]
        )
        log()

        log_stderr("Collecting Host Network Interface Details")
        log("Network Interface Details (Ethtool)")
        jsondata["networking"]["system"]["device"] = {}

        interfaces = set(decode(subprocess.check_output("ip -o a | awk '{print $2}'", shell=True)).splitlines())
        for interface in interfaces:
            if interface.startswith("calic") or interface.endswith("calico"):
                continue

            jsondata["networking"]["system"]["device"][interface] = {}
            log_stderr("Inspecting Network Interface {}".format(interface))
            log("Network Interface Settings - {}".format(interface))
            inspect = self.exec_cmd(["ethtool", interface])
            jsondata["networking"]["system"]["device"][interface]["settings"] = inspect

            log("Network Interface Stats - {}".format(interface))
            inspect = self.exec_cmd(["ethtool", "-S", interface])
            jsondata["networking"]["system"]["device"][interface]["stats"] = inspect

            log("Network Interface Features - {}".format(interface))
            inspect = self.exec_cmd(["ethtool", "-k", interface])
            jsondata["networking"]["system"]["device"][interface]["features"] = inspect

    def overlay_networking(self, net_verbose):
        global networks
        log("Collecting Overlay Network Configuration")
        log("Overlay network configuration")

        jsondata["networking"]["namespaces"] = {}
        net_ids = decode(
            subprocess.check_output(
                ["docker", "network", "ls", "--no-trunc", "--filter", "driver=overlay", "-q"]
            )
        ).splitlines() + ["ingress_sbox"]
        for net in net_ids:
            jsondata["networking"]["namespaces"][net] = {}
            log_stderr("Inspecting Overlay Network {}".format(net))
            log("nnn Network {}".format(net))

            if net != "ingress_sbox":
                nspath = glob.glob("{}/*{}*".format(nsdir, net[:9]))[0]
                inspect = self.exec_cmd(["docker", "network", "inspect", net_verbose, net])
                log_stderr(inspect)
                jsondata["networking"]["namespaces"][net]["inspect"] = inspect
            else:
                nspath = "{}/{}".format(nsdir, net)

            try:
                self.get_ns_info(nspath)
            except Exception as e:
                log_stderr("Error getting network information")
                log_stderr(str(e))
                traceback.print_exc()
            networks += 1

    def bridge_networking(self, net_verbose):
        log("Collecting Bridge Network Configuration")
        log("Bridge network configuration")

        net_ids = decode(
            subprocess.check_output(
                ["docker", "network", "ls", "--no-trunc", "--filter", "driver=bridge", "-q"]
            )
        ).splitlines()
        for net in net_ids:
            jsondata["networking"]["namespaces"][net] = {}
            log_stderr("Inspecting Bridge Network {}".format(net))
            log("nnn Network {}".format(net))
            inspect = self.exec_cmd(["docker", "network", "inspect", net_verbose, net])
            log_stderr(inspect)
            jsondata["networking"]["namespaces"][net]["inspect"] = inspect

    def get_ns_info(self, ns):
        tables = ["filter", "nat", "mangle"]
        jsondata["networking"]["namespaces"][ns] = {}
        jsondata["networking"]["namespaces"][ns]["iptables"] = {}
        jsondata["networking"]["namespaces"][ns]["ip"] = {}
        jsondata["networking"]["namespaces"][ns]["bridges"] = []

        jsondata["networking"]["namespaces"][ns]["ip"]["ipv4"] = self.exec_cmd(
            ["nsenter", "--net={}".format(ns), "ip", "-o", "-4", "address", "show"]
        )
        jsondata["networking"]["namespaces"][ns]["ip"]["ipv4_route"] = self.exec_cmd(
            ["nsenter", "--net={}".format(ns), "ip", "-4", "route", "show"]
        )
        jsondata["networking"]["namespaces"][ns]["ip"]["ipv4_neigh"] = self.exec_cmd(
            ["nsenter", "--net={}".format(ns), "ip", "-4", "neigh", "show"]
        )
        try:
            bridges_raw = json.loads(
                decode(
                    subprocess.check_output(
                        [
                            "nsenter",
                            "--net={}".format(ns),
                            "ip",
                            "-j",
                            "link",
                            "show",
                            "type",
                            "bridge",
                        ]
                    )
                )
            )
        except Exception as e:
            log_stderr(str(e))
            log_stderr("Error getting getting link show type bridge json object")
            traceback.print_exc()
        bridges = []

        for bridge in bridges_raw:
            try:
                if "ifname" in bridge:
                    bridges.append(bridge["ifname"])
            except Exception as e:
                log_stderr(str(e))
                log_stderr("Error adding bridge to json object.")
                traceback.print_exc()
                try:
                    log_stderr(str(bridge))
                except Exception as e:
                    log_stderr(str(e))
                    traceback.print_exc()

        for bridge in bridges:
            if bridge:
                jsondata["networking"]["namespaces"][ns]["bridges"].append(
                    decode(
                        subprocess.check_output(
                            [
                                "nsenter",
                                "--net={}".format(ns),
                                "bridge",
                                "fdb",
                                "show",
                                "br",
                                bridge,
                            ]
                        )
                    ).splitlines()
                )

        for t in tables:
            cmd = "nsenter --net={} iptables -w1 -n -v -L -t {} | grep -v '^$'".format(ns, t)
            jsondata["networking"]["namespaces"][ns]["iptables"][t] = self.parse_iptables(
                self.exec_cmd(cmd, shell=True)
            )

        if shutil.which("ipvsadm"):
            jsondata["networking"]["namespaces"][ns]["ipvsadm"] = self.exec_cmd(
                ["nsenter", "--net={}".format(ns), "ipvsadm", "-l", "-n"]
            )
        else:
            jsondata["networking"]["namespaces"][ns]["ipvsadm"] = []

    @staticmethod
    def parse_iptables(lines):
        chains = {}
        chain = None

        for line in lines:
            if re.match(r"^Chain", line):
                chain = re.match(r"^Chain (.*?)", line).group(1)
                chains[chain] = []
            chains[chain].append(line)

        return chains

    @staticmethod
    def check_ip_overlap(data, netid):
        global ip_overlap
        # Not unpacking this to python for now
        overlap = decode(
            subprocess.check_output(
                'echo "{}" | grep "EndpointIP\\|VIP" | cut -d\':\' -f2 | sort | uniq -c | grep -v "1 "'.format(
                    data
                ),
                shell=True,
            )
        )
        if overlap:
            log("\n\n*** OVERLAP on Network {} ***".format(netid))
            log("{} \n\n".format(overlap))
            ip_overlap += 1
        else:
            log("No overlap")

    def system_metrics(self):
        self.exec_cmd(["hostname"], "hostname")
        self.exec_cmd(["uptime"], "uptime")
        if os.path.exists("/host/usr/lib/os-release"):
            self.exec_cmd("cat /host/usr/lib/os-release", "system_release", shell=True)
        else:
            self.exec_cmd("cat /etc/*release", "system_release", shell=True)
        self.exec_cmd(["cat", "/proc/cmdline"], "container_cmdline")
        self.exec_cmd(["cat", "/proc/version"], "kernel_version")
        self.exec_cmd(["lscpu"], "lscpu")
        self.exec_cmd(["cat", "/proc/meminfo"], "meminfo")
        self.exec_cmd(["cat", "/proc/cgroups"], "system_cgroups")
        self.exec_cmd(["cat", "/proc/self/cgroup"], "running_cgroup")
        self.exec_cmd(["df", "-h"], "df")
        self.exec_cmd(["mount"], "mount")

        if not self.batch:
            self.exec_cmd(["ifconfig"], "ifconfig")
            self.exec_cmd("ps aux | grep docker", "ps_aux_grep_docker", shell=True)
            self.exec_cmd(["netstat", "-npl"], "netstat")
            self.exec_cmd(["sestatus"], "sestatus")
        else:
            jsondata["ifconfig"] = None
            jsondata["ps_aux_grep_docker"] = None
            jsondata["netstat"] = None
            jsondata["sestatus"] = None

        self.exec_cmd(["vmstat", "1", "5"], "vmstat")
        self.exec_cmd(["iostat", "-tkx", "1", "5"], "iostat")
        self.exec_cmd(["dmidecode", "-t", "system"], "dmidecode")

        jsondata["file_max"] = None
        jsondata["docker_pid_limits"] = None

        if os.path.isdir("/host"):
            if os.path.exists("/var/run/docker.pid"):
                docker_pid = open("/var/run/docker.pid").read()
                if os.path.exists("/host/proc/{}/limits".format(docker_pid)):
                    log("dockerd limits")
                    self.exec_cmd(
                        ["cat", "/host/proc/{}/limits".format(docker_pid)], "docker_pid_limits"
                    )
            self.exec_cmd(["cat", "/host/proc/sys/fs/file-max"], "file_max")

    def check_dtr(self):
        jsondata["dtr"] = {}
        jsondata["dtr"]["1.4.3"] = {}
        jsondata["dtr"]["modern"] = {}

        def handle_configs(relative_path, f, k):
            shutil.copy(f, "{}/dtr/{}".format(self.tempdir, relative_path))
            jsondata["dtr"]["1.4.3"][k][os.path.basename(f)] = open(f).readlines()

        dtr_path = "/usr/local/etc/dtr"

        if os.path.isdir(dtr_path):
            log()
            log_header("Docker Trusted Registry 1.4.3")
            os.makedirs("{}/dtr/generatedConfigs".format(self.tempdir))
            os.makedirs("{}/dtr/logs".format(self.tempdir))

            jsondata["dtr"]["1.4.3"]["configs"] = {}
            jsondata["dtr"]["1.4.3"]["generated_configs"] = {}
            jsondata["dtr"]["1.4.3"]["logs"] = {}

            [handle_configs("", f, "configs") for f in glob.glob("{}/*.yml".format(dtr_path))]
            [
                handle_configs("", f, "generated_configs")
                for f in glob.glob("{}/generatedConfigs/*.yml".format(dtr_path))
            ]
            [
                handle_configs("", f, "generated_configs")
                for f in glob.glob("{}/generatedConfigs/*.conf".format(dtr_path))
            ]

            for logfile in glob.glob("{}/logs/*".format(dtr_path)):
                logname = os.path.basename(logfile)
                lines = decode(subprocess.check_output("tail -n {} {}".format(self.loglines, log)))
                with open("{}/dtr/logs/{}".format(self.tempdir, logname)) as f:
                    f.write(lines)
                jsondata["dtr"]["1.4.3"]["logs"][logname] = lines.splitlines()

        else:
            log()
            log("The '/usr/local/etc/dtr' directory does not exist")

            log_header("Docker Trusted Registry 2.0.0 or later")
            jsondata["dtr"]["modern"] = {}
            jsondata["dtr"]["modern"]["network_inspect"] = self.exec_cmd(
                ["docker", "network", "inspect", "dtr-ol"]
            )

            jsondata["dtr"]["modern"]["containers"] = {}

            dtr_containers = decode(
                subprocess.check_output(
                    ["docker", "ps", "-a", "--filter", "name=dtr-", "--format", "{{.Names}}"]
                )
            ).splitlines()

            for c in dtr_containers:
                jsondata["dtr"]["modern"]["containers"][c] = {}
                ulimit = self.exec_cmd(["docker", "exec", c, "sh", "-c", '"ulimit -a"'])
                file_max = self.exec_cmd(["docker", "exec", c, "sh", "-c", '"ulimit -a"'])

                jsondata["dtr"]["modern"]["containers"][c]["ulimit"] = ulimit
                jsondata["dtr"]["modern"]["containers"][c]["file_max"] = file_max

    def system_information(self):
        log_stderr("Collecting Docker/System Information")
        log_header("Docker/System information")
        log("")
        log(self.now)
        log_header("System Info")
        log("sysctl -a")
        self.exec_cmd(["sysctl", "-a"], "system_stats")

        if self.docker:
            log_header("Docker Info")
            self.exec_cmd(["docker", "version"], "docker_version")
            self.exec_cmd(["docker", "info"], "docker_info")
            self.exec_cmd(["docker", "images", "--no-trunc"], "docker_images")
            self.exec_cmd(["docker", "stats", "--all", "--no-stream"], "docker_stats")

            daemon_json = json.loads(open("/etc/docker/daemon.json").read())
            jsondata["docker_daemon_json"] = daemon_json
            log(pprint.pformat(daemon_json, indent=4))

    def cluster_info(self):
        self.get_core_stats()
        jsondata["dead_container_mounts"] = self.exec_cmd(["/bin/find-dead.sh"])
        jsondata["kernel_config"] = self.exec_cmd(["/check-config.sh"])

    def get_core_stats(self):
        output = self.exec_cmd(["/bin/core-counts.sh"])
        jsondata["cluster_info"] = {}
        cats = output[1:]
        for cat in cats:
            fields = cat.split()
            core_count = fields[2]
            if int(core_count) > 0:
                average_cores = fields[3]
                max_cores = fields[4]
                min_cores = fields[5]
            else:
                average_cores, max_cores, min_cores = 0, 0, 0
            jsondata["cluster_info"][0] = {}
            jsondata["cluster_info"][0]["core_count"] = core_count
            jsondata["cluster_info"][0]["average_cores"] = average_cores
            jsondata["cluster_info"][0]["max_cores"] = max_cores
            jsondata["cluster_info"][0]["min_cores"] = min_cores

    def get_docker_events(self):
        log_stderr("Gathering Docker Event logs")
        jsondata["docker_events"] = {}
        output = ""
        event_date_filter = []
        if "--since" in self.date_filter_docker:
            i = self.date_filter_docker.index("--since")
            event_date_filter += self.date_filter_docker[i:i+2]
        else:
            event_date_filter += ["--since", "24h"]

        if "--until" in self.date_filter_docker:
            i = self.date_filter_docker.index("--until")
            event_date_filter += self.date_filter_docker[i:i+2]
        else:
            event_date_filter += ["--until", "1s"]
        cmd = ["docker", "events"] + event_date_filter
        try:
            output = decode(subprocess.check_output(cmd,stderr=subprocess.STDOUT))
        except BaseException as e:
             log_stderr("Error getting docker events: {}".format(e))
        with open("{}/events.log".format(self.tempdir), "w") as f:
            f.write(output)
        jsondata["docker_events"] = output.splitlines()

    def get_docker_logs(self):
        log_stderr("Gathering Docker Daemon logs")
        if os.path.isfile("/var/log/docker.log"):
            # Sucking the whole file in in Python is risky, and re-implementing
            # a bunch of seeks when we can shell out isn't ideal. Just use optimized
            # utilities
            output = decode(
                subprocess.check_output(["tail", "-n", self.loglines, "/var/log/docker/log"])
            )
            with open("{}/var_log_daemon.log".format(self.tempdir), "w") as f:
                f.write(output)
            jsondata["var_log_daemon"] = output.splitlines()
        cmd = ["journalctl", "--no-pager", "-n", self.loglines, "-o", "json", "-u",
               "docker.service"] + self.date_filter_jctl
        output = subprocess.check_output(
            cmd
        )
        log_output = decode(
            subprocess.check_output(
                ["jq", "-r", 'select(has("CONTAINER_ID") | not ) .MESSAGE'], input=output
            )
        )
        # We have to pass bytes into input=, so delay decoding
        output = decode(output)

        with open("{}/journalctl_daemon.log".format(self.tempdir), "w") as f:
            f.write(log_output)

        json_append = []
        for line in output.splitlines():
            j = json.loads(line)
            if "CONTAINER_ID" not in j:
                json_append.append(j["MESSAGE"])

        jsondata["journalctl_daemon"] = json_append

    def get_system_logs(self):
        log_stderr("Gathering Kernel Messages")
        jsondata_kernel = []
        jsondata_dmesg = []
        if shutil.which("journalctl"):
            cmd = ["journalctl", "-n", self.loglines, "-k"] + self.date_filter_jctl
            output = decode(subprocess.check_output(cmd))
            with open("{}/journalctl_kernel.log".format(self.tempdir), "w") as f:
                f.write(output)
            with open("{}/dmesg.txt".format(self.tempdir), "w") as f:
                f.write(output)

            cmd += ["-o", "json"]
            jsonoutput = decode(subprocess.check_output(cmd))
            jsondata_kernel = [json.loads(line) for line in jsonoutput.splitlines()]
            jsondata_dmesg = jsondata_kernel

        else:
            output = decode(
                subprocess.check_output("dmesg -T | tail -n {}".format(self.loglines), shell=True)
            )
            with open("{}/dmesg.txt".format(self.tempdir), "w") as f:
                f.write(output)
            jsondata_dmesg = output.splitlines()
        jsondata["journalctl_kernel"] = jsondata_kernel
        jsondata["dmesg"] = jsondata_dmesg

    def get_journalctl_logs(self, service_unit, logfile_name):
        log_stderr("Gathering Journalctl logs")
        logs = []
        if shutil.which("journalctl"):
            cmd = ["journalctl", "--no-pager", "-n", self.loglines, "-u", service_unit] + self.date_filter_jctl
            output = decode(subprocess.check_output(cmd))
            with open("{}/{}".format(self.tempdir, logfile_name), "w") as f:
                f.write(output)
            logs = output.splitlines()

        return logs

    def get_cri_dockerd_logs(self):
        log_stderr("Gathering CRI-dockerd logs")

        # Get cri-dockerd logs
        jsondata_cri_dockerd_service_logs = self.get_journalctl_logs("cri-dockerd-mke.service", "journalctl_cri_dockerd_service.log")
        jsondata_cri_dockerd_socket_logs = self.get_journalctl_logs("cri-dockerd-mke.socket", "journalctl_cri_dockerd_socket.log")

        jsondata["journalctl_cri_dockerd"] = {}
        jsondata["journalctl_cri_dockerd"]["service"] = jsondata_cri_dockerd_service_logs
        jsondata["journalctl_cri_dockerd"]["socket"] = jsondata_cri_dockerd_socket_logs

    def get_containerd_logs(self):
        log_stderr("Gathering Containerd Logs")

        # Get containderd logs
        jsondata_containerd_logs = self.get_journalctl_logs("containerd", "journalctl_containerd.log")

        # Get containerd config
        jsondata_containerd_config = []
        try:
            shutil.copy("/etc/containerd/config.toml", "{}/containerd_config.toml".format(self.tempdir))
            jsondata_containerd_config = open("{}/containerd_config.toml".format(self.tempdir)).read().splitlines()

        except BaseException as e:
            log_stderr("Failed to copy containerd config.")

        jsondata["containerd"] = {}
        jsondata["containerd"]["logs"] = jsondata_containerd_logs
        jsondata["containerd"]["config"] = jsondata_containerd_config

    def get_stacktraces(self):
        def update_stack_traces():
            files = glob.glob("/run/docker/goroutine-stacks-*log")
            return files

        log_stderr("Gathering daemon stack trace")
        old_stack_traces = update_stack_traces()
        docker_pid = open("/var/run/docker.pid").read()
        subprocess.call(["kill", "-s", "USR1", docker_pid])

        stack_traces = update_stack_traces()
        count = 0
        while stack_traces == old_stack_traces and count < 10:
            time.sleep(1)
            count += 1
            stack_traces = update_stack_traces()

        if len(stack_traces) > 0:
            jsondata["daemon_stack_trace"] = open(stack_traces[0]).read().splitlines()
            shutil.move(stack_traces[0], "{}/daemon-stack-trace.log".format(self.tempdir))

    def get_container_info(self):
        log_stderr("Gathering UCP/DTR/Kubernetes System Container Info")

        jsondata["container_info"] = {}
        containers = []
        filters = [
            "name=ucp-",
            "name=dtr-",
            "label=io.kubernetes.pod.namespace=kube-system",
            "label=io.kubernetes.pod.namespace=ingress-nginx",
            "label=io.kubernetes.pod.namespace=docker",
            "label=type=com.docker.interlock.core.controller",
            "label=type=com.docker.interlock.core.extension",
            "label=type=com.docker.interlock.core.proxy",
            "label=type=com.docker.interlock.core.config",
        ]

        for f in filters:
            containers.append(
                decode(
                    subprocess.check_output(
                        ["docker", "ps", "-a", "--filter", f, "--format", "{{.Names}}"]
                    )
                ).splitlines()
            )

        # flatten it
        containers = [c for sub in containers for c in sub]
        containers = set(containers)

        for c in sorted(containers):
            log_stderr("    Collecting Logs for container {}".format(c))
            if c == "ucp-controller":
                jsondata["ucp_controller_diag"] = {}

                diagpath = "{}/diag".format(self.tempdir)
                if not os.path.exists(diagpath):
                    os.makedirs(diagpath)

                cmd = "docker exec {} tar -C /tmp/diag -cf - . | tar -C {} -xf -".format(c, diagpath)
                subprocess.call(cmd, shell=True)

                jsondata["container_info"]["ucp-controller"] = {}

                diags = glob.glob("{}/*".format(diagpath))
                for diag in diags:
                    if not self.stacktraces and diag.__contains__("goroutine-stacks"):
                        os.remove(diag)
                        continue
                    jsondata["ucp_controller_diag"][os.path.basename(diag)] = open(
                        diag
                    ).readlines()

            if c == "ucp-auth-store":
                log_stderr("Collecting RethinkDB Status")
                node_addr = decode(
                    subprocess.check_output(["docker", "info", "--format", "{{.Swarm.NodeAddr}}"])
                ).strip()
                image_name = decode(
                    subprocess.check_output(
                        [
                            "docker",
                            "service",
                            "inspect",
                            "ucp-auth-api",
                            "--format",
                            "{{.Spec.TaskTemplate.ContainerSpec.Image}}",
                        ]
                    )
                ).strip()

                network = decode(
                    subprocess.check_output(
                        [
                            "docker",
                            "network",
                            "ls",
                            "--filter",
                            "name=ucp-bridge",
                            "--format",
                            "{{.Name}}",
                        ]
                    )
                ).strip()

                if "ucp-bridge" in network:
                    network = "ucp-bridge"
                else:
                    log_stderr("ucp-bridge network doesn't exist, using default bridge")
                    network = "bridge"

                rethink_status = decode(
                    subprocess.check_output(
                        [
                            "docker",
                            "run",
                            "--rm",
                            "--network",
                            network,
                            "-v",
                            "ucp-auth-store-certs:/tls",
                            image_name,
                            "--db-addr={}:12383".format(node_addr),
                            "db-status",
                        ]
                    )
                )
                with open("{}/rethinkdb-status.json".format(self.tempdir), "w") as f:
                    f.write(rethink_status)

                rethink_json = json.loads(rethink_status)
                jsondata["rethink_status"] = rethink_json

            if c == "ucp-kv":
                log_stderr("Collecting etcd Endpoint(s) Health")
                etcd_endpoint_health = decode(
                    subprocess.check_output(
                        [
                            "docker",
                            "exec",
                            "ucp-kv",
                            "sh",
                            "-c",
                            "etcdctl --cluster=true endpoint health -w table 2>/dev/null"
                        ]
                    )
                )
                log_stderr("Collecting etcd Endpoint(s) Status")
                etcd_endpoint_status = decode(
                    subprocess.check_output(
                        [
                            "docker",
                            "exec",
                            "ucp-kv",
                            "sh",
                            "-c",
                            "etcdctl --cluster=true endpoint status -w table 2>dev/null"
                        ]
                    )
                )
                log_stderr("Collecting Cluster Health")
                cluster_health = decode(
                    subprocess.check_output(
                        [
                            "docker",
                            "exec",
                            "-e",
                            "ETCDCTL_API=2",
                            "ucp-kv",
                            "etcdctl",
                            "--endpoint",
                            "https://127.0.0.1:2379",
                            "cluster-health"
                        ]
                    )
                )
                log_stderr("Collecting etcd Member List")
                member_list = decode(
                    subprocess.check_output(
                        [
                            "docker",
                            "exec",
                            "-e",
                            "ETCDCTL_API=2",
                            "ucp-kv",
                            "etcdctl",
                            "--endpoint",
                            "https://127.0.0.1:2379",
                            "member",
                            "list"
                        ]
                    )
                )

                with open("{}/etcd-status.txt".format(self.tempdir), "w") as f:
                    f.write(
                        etcd_endpoint_health + "\n" + etcd_endpoint_status + "\n" + cluster_health + "\n" + member_list)

            if str.startswith(c, "dtr-rethinkdb"):
                try:
                    replica_id = subprocess.check_output(
                        "docker ps -lf name='^/dtr-rethinkdb-.{12}$' --format '{{.Names}}' | cut -d- -f3",
                        shell=True).decode().strip()

                    rethinkcli = ' '.join(["docker run",
                                           "-i --rm --net dtr-ol",
                                           "-e DTR_REPLICA_ID={}".format(replica_id),
                                           "-v dtr-ca-{}:/ca".format(replica_id),
                                           "mirantis/rethinkcli:{}".format(os.getenv('RETHINKCLI_NI_VERSION')),
                                           "non-interactive"
                                           ])

                    log_stderr("Collecting RethinkDB Table Status")
                    rethinkdb_table_status = subprocess.check_output(
                        ("echo \"r.db('rethinkdb').table('table_status')\" |" + rethinkcli), shell=True).decode()

                    rethinkdb_table_status_json = json.loads(rethinkdb_table_status)

                    with open("{}/rethinkdb-table-status.json".format(self.tempdir), "w") as f:
                        json.dump(rethinkdb_table_status_json, f, indent=4)

                    jsondata["rethinkdb_table_status"] = rethinkdb_table_status_json

                    log_stderr("Collecting RethinkDB Properties ha.json")
                    ha_json = subprocess.check_output(
                        ("echo \"r.db('dtr2').table('properties').get('ha.json')\" |" + rethinkcli),
                        shell=True).decode()

                    rethinkdb_ha_json = json.loads(ha_json)

                    with open("{}/rethinkdb-ha.json".format(self.tempdir), "w") as f:
                        json.dump(json.loads(rethinkdb_ha_json["value"]), f, indent=4)

                    jsondata["ha_json"] = rethinkdb_ha_json

                    log_stderr("Collecting RethinkDB Properties storage.yml")
                    rethinkdb_storage = subprocess.check_output(
                        ("echo \"r.db('dtr2').table('properties').get('storage.yml')\" |" + rethinkcli),
                        shell=True).decode()

                    rethinkdb_storage_json = json.loads(rethinkdb_storage)

                    with open("{}/rethinkdb-storage.yaml".format(self.tempdir), "w") as f:
                        yaml.dump(yaml.safe_load(rethinkdb_storage_json["value"]), f, indent=4)

                    jsondata["rethinkdb_storage"] = rethinkdb_storage_json

                    log_stderr("Collecting RethinkDB Jobrunner Workers")
                    rethinkdb_jobrunner_workers = subprocess.check_output(
                        ("echo \"r.db('jobrunner').table('workers')\" |" + rethinkcli), shell=True).decode()

                    rethinkdb_jobrunner_workers_json = json.loads(rethinkdb_jobrunner_workers)

                    with open("{}/rethinkdb-jobrunner-workers.json".format(self.tempdir), "w") as f:
                        json.dump(rethinkdb_jobrunner_workers_json, f, indent=4)

                    jsondata["rethinkdb_jobrunner_workers"] = rethinkdb_jobrunner_workers_json

                except subprocess.CalledProcessError as e:
                    log_stderr("Couldn't capture MSR RethinkDB details in Support Bundle: {}".format(e.output))

            with open("{}/logs/{}.log".format(self.tempdir, c), "w") as f:
                log(c, f)

                jsondata["container_info"][c] = {}
                jsondata["container_info"][c]["logs"] = []
                jsondata["container_info"][c]["inspect"] = None
                output = ""

                if c == "ucp-controller" and not self.audit:
                    cmd = ["docker", "logs", "--timestamps", "--tail", self.loglines, c] + self.date_filter_docker
                    output = decode(
                        subprocess.check_output(
                            cmd,
                            stderr=subprocess.STDOUT,
                        )
                    )
                    # Filter out lines which contain auditID if this was run with -a
                    output = output.splitlines()
                    output = [o for o in output if "auditID" not in o]
                    output = "\n".join(output)
                else:
                    try:
                        cmd = ["docker", "logs", "--timestamps", "--tail", self.loglines, c] + self.date_filter_docker
                        output = decode(
                            subprocess.check_output(
                                cmd,
                                stderr=subprocess.STDOUT,
                            )
                        )
                    except subprocess.CalledProcessError:
                        # There's a potential timing issue where some containers, like 'ucp-kubelet'
                        # may disappear while running. Also a bug in the original code, but it was
                        # not run with -x
                        pass

                log(output, f)
                jsondata["container_info"][c]["logs"] = output.splitlines()

            with open("{}/inspect/{}.txt".format(self.tempdir, c), "w") as f:
                log_stderr("    Collecting 'docker inspect' output for container {}".format(c))

                try:
                    log(decode(subprocess.check_output(["docker", "inspect", c])), f)
                    inspect_json = json.loads(
                        decode(
                            subprocess.check_output(
                                ["docker", "inspect", c, "--format", "{{json .}}"]
                            )
                        )
                    )
                    jsondata["container_info"][c]["inspect"] = inspect_json
                except subprocess.CalledProcessError:
                    # There's a potential timing issue where some containers, like 'ucp-kubelet'
                    # may disappear while running. Also a bug in the original code, but it was
                    # not run with -x
                    pass

    def makedirs(self):
        try:
            shutil.rmtree(self.tempdir)
        except FileNotFoundError:
            # Doesn't already exist, probably not run yet
            pass
        os.makedirs(self.tempdir)

        if self.docker:
            os.makedirs("{}/inspect".format(self.tempdir))
            os.makedirs("{}/logs".format(self.tempdir))

    def shim_logs(self):
        shimpath = "{}/shimlogs".format(self.tempdir)
        os.makedirs(shimpath)

        files = glob.glob("/host/tmp/ucp-*-shim-debug.log")

        jsondata["shim_logs"] = {}

        for f in files:
            destination = re.sub(
                r"ucp-([a-z]+)-[a-zA-Z0-9]+-shim-debug.log",
                r"ucp-\1-shim-debug.log",
                os.path.basename(f),
            )
            shutil.copy(f, "{}/{}".format(shimpath, destination))
            with open(f, "r") as g:
                lines = g.readlines()
                jsondata["shim_logs"][os.path.basename(f)] = lines

    def get_xfs_info(self):
        log_header("xfs_info")
        log_stderr("Collecting xfs_info")
        host_dir = "/host"
        xfs_locations = [
            "/var/lib/docker",
            "/var/lib/containerd",
            "/var/lib/kubelet"
        ]
        jsondata["xfs_info"] = {}
        for loc in xfs_locations:
            log("xfs_info {}".format(loc))
            container_location = host_dir + loc
            jsondata["xfs_info"][container_location] = self.exec_cmd(["xfs_info", container_location])

    def parse_args(self):
        parser = argparse.ArgumentParser(
            prog="dsinfo", description="Collect support bundles for Mirantis UCP/MKE"
        )

        parser.add_argument(
            "-b",
            "--batch",
            dest="batch",
            default=False,
            action="store_true",
            help="Run in batch mode",
        )

        parser.add_argument(
            "-d",
            "--docker",
            dest="docker",
            default=True,
            action="store_true",
            help="When run with [-d|--debug], the script will also perform Docker API calls",
        )

        parser.add_argument(
            "-a",
            "--audit",
            dest="audit",
            default=False,
            action="store_true",
            help="When run with [-a|--audit], the script will include audit logs from "
                 "the UCP controller container",
        )

        parser.add_argument(
            "-s",
            "--ssd",
            dest="ssd",
            default=False,
            action="store_true",
            help="When run with [-s|-ssd], the script will theck the SSD control plane",
        )

        parser.add_argument(
            "-l",
            "--loglines",
            dest="loglines",
            default="10000",
            type=str,
            help="[-l|--logines] specifies the number of log lines to capture",
        )

        parser.add_argument(
            "-t",
            "--td",
            dest="tempdir",
            default=tempdir_default,
            type=str,
            help="[-t|--td] specifies the tempdir to use",
        )

        parser.add_argument(
            "-o",
            "--output",
            dest="tarfile",
            default="/dsinfo.tar.gz",
            type=str,
            help="[-o|--output] specifies the output tarball",
        )

        parser.add_argument(
            "-g",
            "--goroutine",
            dest="stack_trace",
            default=False,
            action="store_true",
            help="When run with [-g|--goroutine], the script will retrieve goroutine stacktraces",
        )

        parser.add_argument(
            "--since",
            dest="since",
            type=validate_date,
            help="Provide logs since a date/time in the format '2022-01-01 12:00:00'",
        )

        parser.add_argument(
            "--until",
            dest="until",
            type=validate_date,
            help="Provide logs until a date/time in the format '2022-01-01 12:00:00'",
        )

        args = parser.parse_args()

        self.batch = args.batch
        self.docker = args.docker
        self.audit = args.audit
        self.ssd = args.ssd
        self.loglines = args.loglines
        self.tempdir = args.tempdir
        self.tarfile = args.tarfile
        self.stacktraces = args.stack_trace
        self.date_filter_docker = []
        self.date_filter_jctl = []
        if args.since:
            self.date_filter_docker += ["--since", args.since.strftime("%Y-%m-%dT%H:%M:%S")]
            self.date_filter_jctl += ["--since", args.since.strftime("%Y-%m-%d %H:%M:%S")]
        if args.until:
            self.date_filter_docker += ["--until", args.until.strftime("%Y-%m-%dT%H:%M:%S")]
            self.date_filter_jctl += ["--until", args.until.strftime("%Y-%m-%d %H:%M:%S")]


if __name__ == "__main__":
    if os.getuid() != 0:
        print("This script must be run as root or sudo!")
        sys.exit(1)
    d = DSinfo()
    default_handle = open(output_default, "w")
    try:
        d.run()
        d.finish()

    except BaseException as e:
        traceback.print_exc()
        d.finish()
