#!/bin/sh

# If it doesn't already exist, create a subdirectory of /data that etcd will
# use for its data directory. This leaves the original data directory structure
# under /data untouched; this ensures that if UCP has to roll back to an older
# version of etcd, that older version will be able to use the original data
# dir at /data and won't see the newer v3 data, which causes it to crash.
if [ ! -d /data/datav3 ]; then
    cp -r /data /data/datav3
fi

# etcd requires that the data directory have restricted permissions.
chmod 700 /data/datav3

# For the purposes of this script, a "member" is the single key=value pair
# of the name of a member and its advertised peer URL.
list_members() {
    ENDPOINTS="$1"
    export ETCDCTL_API=3
    unset ETCDCTL_CA_FILE ETCDCTL_CERT_FILE ETCDCTL_KEY_FILE
    # As of version 3.5 etcd offers a linearizable guarantee for the member list
    # endpoint. If etcd does not have quorum it will hang. This will always
    # happen when members are added to a single-node etcd cluster, which is
    # how MKE bootstraps etcd during installation.
    # We get the members list via curl so that we may set linearizable=false.
    # The result is processed into one key=value pair per line to match
    # the format used by the --initial-cluster argument
    # Background: https://mirantis.jira.com/browse/MKE-9187
    for ENDPOINT in ${ENDPOINTS//,/ }; do
      RES=$(curl -s -m 10 --cacert "$ETCDCTL_CACERT" --cert "$ETCDCTL_CERT" --key "$ETCDCTL_KEY" \
      https://"$ENDPOINT"/v3/cluster/member/list -X POST -d '{"linearizable":false}') && break
    done
    echo "${RES}" | \
    jq -r ".members | .[] | [.name, .peerURLs[0]] | @csv" | \
    tr -d '"' | \
    awk -F ',' '{ print $1 "=" $2 }'
}

# If there is no member directory, then that means that this server has not yet
# been initialized. If there is no wal (write-ahead-log) directory then etcd
# has not been fully configured with member/cluster ID info. In either case
# this etcd instance needs to either join an existing cluster OR initialize
# a new cluster.
EXTRA_ARGS=""
if [ ! -d /data/datav3/member/wal ]; then
    # We should have extra args file contains extra environment variables which
    # indicate whether we should initialize a new cluster or join an existing
    # cluster. This will also contain environment variables which indicate the
    # local member name and peer addr. The expected variables are:
    #   INITIAL_CLUSTER_STATE
    #   LOCAL_NAME
    #   LOCAL_PEER_URL
    #   ENDPOINTS
    source /data/extra.args
    if [ "$INITIAL_CLUSTER_STATE" = "new" ]; then
        # This is a new cluster! We construct our extra args from our local
        # name and peer URL.
        echo "Initializing new etcd cluster ..."
        EXTRA_ARGS="--initial-cluster-state new"
        EXTRA_ARGS="$EXTRA_ARGS --initial-cluster $LOCAL_NAME=$LOCAL_PEER_URL"
        EXTRA_ARGS="$EXTRA_ARGS --initial-advertise-peer-urls $LOCAL_PEER_URL"
    else
        # We are joining an existing cluster! We must query the existing
        # servers in the cluster for the list of members until the local peer
        # URL appears in the members list (without a name because it has not
        # been set yet. At that point, we can set the initial cluster argument
        # from the listed members.
        echo "Joining existing etcd cluster ..."
        echo "Waiting for local peer $LOCAL_PEER_URL to appear in etcd cluster membership"
        until list_members "$ENDPOINTS" | grep -q '^='"$LOCAL_PEER_URL"'$'; do
            echo "..."
            sleep 5
        done

        INITIAL_CLUSTER=""
        for MEMBER in $(list_members "$ENDPOINTS"); do
            if [ "$MEMBER" = "=$LOCAL_PEER_URL" ]; then
                MEMBER="$LOCAL_NAME=$LOCAL_PEER_URL"
            fi
            if [ -z "$INITIAL_CLUSTER" ]; then
                INITIAL_CLUSTER="$MEMBER"
            else
                INITIAL_CLUSTER="$INITIAL_CLUSTER,$MEMBER"
            fi
        done

        EXTRA_ARGS="--initial-cluster-state existing"
        EXTRA_ARGS="$EXTRA_ARGS --initial-cluster $INITIAL_CLUSTER"
        EXTRA_ARGS="$EXTRA_ARGS --initial-advertise-peer-urls $LOCAL_PEER_URL"
    fi
    # exposes grpc_server_handling_seconds histogram metrics.
    EXTRA_ARGS="$EXTRA_ARGS --metrics=extensive"
fi

EXTRA_ARGS="$EXTRA_ARGS --logger=zap"
ETCD_ARGS="$@ $EXTRA_ARGS"

# Make sure we log all of the arguments.
echo "Starting etcd server with arguments:" $ETCD_ARGS

exec /bin/etcd $ETCD_ARGS
