#!/bin/sh

# Copy config files to writeable volume
cp /etc/prometheus/prometheus.yml /prometheus_config/prometheus.yml

if [ -n "$ENABLE_KUBE_PROXY_METRICS" ]; then
  if ! grep -q "kube-proxy-nodes" /prometheus_config/prometheus.yml; then
    cat << EOF >> /prometheus_config/prometheus.yml
  - job_name: kube-proxy-nodes
    bearer_token_file: /var/run/secrets/kubernetes.io/serviceaccount/token
    kubernetes_sd_configs:
      - role: node
    relabel_configs:
      - action: labelmap
        regex: __meta_kubernetes_node_label_(.+)
      - replacement: kubernetes.default.svc:443
        target_label: __address__
      - regex: (.+)
        replacement: /api/v1/nodes/\$1:10249/proxy/metrics
        source_labels:
          - __meta_kubernetes_node_name
        target_label: __metrics_path__
    scheme: https
    tls_config:
      ca_file: /var/run/secrets/kubernetes.io/serviceaccount/ca.crt

EOF
  fi

fi

if [ -n "$ENABLE_NODE_EXPORTER" ]; then

  if ! grep -q "node-exporter" /prometheus_config/prometheus.yml; then
    # Add node-exporter configuration to the prometheus config
    cat << EOF >> /prometheus_config/prometheus.yml
  - job_name: 'node-exporter'
    tls_config:
        ca_file: /node_certs/ca.pem
        cert_file: /node_certs/cert.pem
        key_file: /node_certs/key.pem
    scheme: https
    file_sd_configs:
      - files:
        - /node_targets/node_exporter_targets.yml

EOF

  fi

fi

if [ -n "$ENABLE_CADVISOR_METRICS" ]; then
  if ! grep -q "ucp-cadvisor" /prometheus_config/prometheus.yml; then
    cat << EOF >> /prometheus_config/prometheus.yml
  - job_name: ucp-cadvisor
    kubernetes_sd_configs:
      - role: endpoints
    relabel_configs:
      - source_labels: [__meta_kubernetes_endpoints_name]
        regex: 'ucp-cadvisor'
        action: keep
    metric_relabel_configs:
      # rename container_label_io_kubernetes_pod_namespace to namespace
      - action: replace
        source_labels: [container_label_io_kubernetes_pod_namespace]
        target_label: namespace
      - action: labeldrop
        regex: container_label_io_kubernetes_pod_namespace
      # rename container_label_io_kubernetes_pod_name to pod
      - action: replace
        source_labels: [container_label_io_kubernetes_pod_name]
        target_label: pod
      - action: labeldrop
        regex: container_label_io_kubernetes_pod_name
      # rename container_label_io_kubernetes_container_name to container
      - action: replace
        source_labels: [container_label_io_kubernetes_container_name]
        target_label: container
      - action: labeldrop
        regex: container_label_io_kubernetes_container_name

EOF
  fi

fi

if [ -n "$ENABLE_ALERTMANAGER" ]; then

  cp /etc/prometheus/prometheus_alert_rules.yml /prometheus_config/prometheus_alert_rules.yml

  if ! grep -q "blackbox-ucp" /prometheus_config/prometheus.yml; then
    # Add blackbox and alertmanager configuration to the prometheus config
    cat << EOF >> /prometheus_config/prometheus.yml
  - job_name: 'blackbox-ucp'
    scrape_interval: 15s
    scrape_timeout: 10s
    metrics_path: /probe
    params:
      module: [http_2xx]
    static_configs:
      - targets:
$BLACKBOX_TARGETS
    relabel_configs:
      - source_labels: [__address__]
        target_label: __param_target
      - source_labels: [__param_target]
        target_label: instance
      - target_label: __address__
        replacement: localhost:9115

rule_files:
  - /prometheus_config/prometheus_alert_rules.yml

alerting:
  alertmanagers:
    - static_configs:
        - targets:
              - localhost:9093

EOF

  sed -e "s#\$BLACKBOX_TARGETS#$BLACKBOX_TARGETS#g" \
      -i"" /prometheus_config/prometheus.yml
  fi

fi

# Replace the placeholders in prometheus.yml with the real server name
# we want to use when making SSL requests to controllers, as well as other
# variables.
sed -e "s/\$SCRAPE_INTERVAL/$SCRAPE_INTERVAL/g" \
    -e "s/\$TLS_SERVER_NAME/$TLS_SERVER_NAME/g" \
    -e "s/\$ETCD_TARGETS/$ETCD_TARGETS/g" \
    -e "s/\$KUBE_CONTROLLER_MANAGER_METRICS_PROXY_PORT/$KUBE_CONTROLLER_MANAGER_METRICS_PROXY_PORT/g" \
    -i"" /prometheus_config/prometheus.yml

# if there are multiple targets, they will be separated with \\n,
# so, we must replace them with newlines
sed -e 's/\\n/\'$'\n/g' \
    -i"" /prometheus_config/prometheus.yml

exec /bin/prometheus "$@"
