#!/bin/sh

HOST_IP=$(route -n | awk '/UG[ \t]/{print $2}')
SF_NOTIFIER_URL="http://$HOST_IP:$SF_NOTIFIER_PORT/hook"

sed -e "s#\$SF_NOTIFIER_URL#$SF_NOTIFIER_URL#g" \
    -i"" /etc/prometheus/alertmanager.yml

if [ -n "$ENABLE_ALERTMANAGER" ] && ! grep -q "blackbox-ucp" /etc/prometheus/prometheus_swarm_only.yml; then
  cat << EOF >> /etc/prometheus/prometheus_swarm_only.yml

  - job_name: 'blackbox-ucp'
    scrape_interval: 15s
    scrape_timeout: 10s
    metrics_path: /probe
    params:
      module: [http_2xx]
    static_configs:
      - targets:
$BLACKBOX_TARGETS
    relabel_configs:
      - source_labels: [__address__]
        target_label: __param_target
      - source_labels: [__param_target]
        target_label: instance
      - target_label: __address__
        replacement: localhost:9115

rule_files:
  - /etc/prometheus/prometheus_alert_rules.yml

alerting:
  alertmanagers:
    - static_configs:
        - targets:
              - localhost:9093
EOF

  sed -e "s#\$BLACKBOX_TARGETS#$BLACKBOX_TARGETS#g" \
      -i"" /etc/prometheus/prometheus_swarm_only.yml
fi

# Replace the placeholders in prometheus.yml with the real server name
# we want to use when making SSL requests to controllers, as well as other
# variables.
sed -e "s/\$SCRAPE_INTERVAL/$SCRAPE_INTERVAL/g" \
    -e "s/\$TLS_SERVER_NAME/$TLS_SERVER_NAME/g" \
    -e "s/\$ETCD_TARGETS/$ETCD_TARGETS/g" \
    -e "s/\$RETHINK_EXPORTER_TARGETS/$RETHINK_EXPORTER_TARGETS/g" \
    -i"" /etc/prometheus/prometheus_swarm_only.yml

# if there are multiple targets, they will be separated with \\n,
# so, we must replace them with newlines
sed -e 's/\\n/\'$'\n/g' \
    -i"" /etc/prometheus/prometheus_swarm_only.yml

PROXY_PID=
PROMETHEUS_PID=
ALERTMANAGER_PID=
BLACKBOX_PID=

# If either the proxy or prometheus servers die, we should kill the other and
# exit.
childHandler() {
    if kill -0 $PROXY_PID; then
        # The proxy is still alive; kill it.
        kill -9 $PROXY_PID
    fi
    if kill -0 $PROMETHEUS_PID; then
        # Prometheus is still alive; kill it.
        kill -9 $PROMETHEUS_PID
    fi
    if kill -0 $ALERTMANAGER_PID; then
        # Alertmanager is still alive; kill it.
        kill -9 $ALERTMANAGER_PID
    fi
    if kill -0 $BLACKBOX_PID; then
        # Blackbox is still alive; kill it.
        kill -9 $BLACKBOX_PID
    fi
}

trap childHandler CHLD

PROXY_ARGS="-port=12387"
if [ -n "$DISABLE_HTTP2" ]; then
  echo "Running swarm-mode ucp-metrics proxy with HTTP2 disabled"
  PROXY_ARGS="$PROXY_ARGS --disable-http2"
fi

/bin/proxy ${PROXY_ARGS} &
PROXY_PID=$!

echo "started proxy server with PID $PROXY_PID"

/bin/prometheus "$@" &
PROMETHEUS_PID=$!

echo "started prometheus server with PID $PROMETHEUS_PID"

if [ -n "$ENABLE_ALERTMANAGER" ]; then
  /bin/blackbox_exporter --config.file=/etc/prometheus/blackbox.yml &
  BLACKBOX_PID=$!
  echo "started blackbox with PID $BLACKBOX_PID"

  /bin/alertmanager --config.file=/etc/prometheus/alertmanager.yml &
  ALERTMANAGER_PID=$!
  echo "started alertmanager with PID $ALERTMANAGER_PID"
fi

# Wait for both the proxy and prometheus servers to exit.
wait $PROXY_PID $PROMETHEUS_PID
# Wait for alertmanager (and blackbox) to exit if it was enabled
[ -n "$ALERTMANAGER_PID" ]; wait $ALERTMANAGER_PID $BLACKBOX_PID
