#!/bin/bash
#
# Collects info about CPU cores from all nodes
# Assumes jq is available
#
# https://mirantis.jira.com/browse/ENGORC-7588

declare resp=''

# avoid divide by 0 when there may not be any nodes of a given type
# however min and max in this case is undefined so null is ok for those
get_counts_for_all_nodes() {
        resp=$(docker node inspect `docker node ls -q`|jq -c -r '[.[]|.Description.Resources.NanoCPUs/1000000000|floor]|[([length,0]|max),([add,0]|max),([(([add,0]|max)/([length,1]|max))]|.[0]*100|[floor]|.[0]/100),min,max]|@csv')
}

get_counts_for_all_nodes_with_given_os() {
        resp=$(docker node inspect `docker node ls -q`|jq -c -r '[.[]|select(.Description.Platform.OS == "'"${1}"'")|.Description.Resources.NanoCPUs/1000000000|floor]|[([length,0]|max), ([add,0]|max), ([(([add,0]|max)/([length,1]|max))]|.[0]*100|[floor]|.[0]/100), min, max]|@csv')
}

get_counts_for_all_nodes_with_given_role() {
        resp=$(docker node inspect `docker node ls -q`|jq -c -r '[.[]|select(.Spec.Role == "'"${1}"'")|.Description.Resources.NanoCPUs/1000000000|floor]|[([length,0]|max), ([add,0]|max), ([(([add,0]|max)/([length,1]|max))]|.[0]*100|[floor]|.[0]/100), min, max]|@csv')
}

get_counts_for_all_nodes_with_given_os_and_role() {
        resp=$(docker node inspect `docker node ls -q`|jq -c -r '[.[]|select(.Description.Platform.OS == "'"${1}"'" and .Spec.Role == "'"${2}"'")|.Description.Resources.NanoCPUs/1000000000|floor]|[([length,0]|max), ([add,0]|max), ([(([add,0]|max)/([length,1]|max))]|.[0]*100|[floor]|.[0]/100), min, max]|@csv')
}

emit_stats() {
        output="${1},$resp"
        echo "$output"|tr , '   '
}

get_and_emit_counts() {
        if [ "$1" == "all" ] && [ "$2" == "all" ]; then
               get_counts_for_all_nodes
               emit_stats "All nodes"
        elif [ "$1" != "all" ] && [ "$2" != "all" ]; then
               get_counts_for_all_nodes_with_given_os_and_role "$1" "$2"
               emit_stats "${1} ${2}"
        elif [ "$1" != "all" ]; then
               get_counts_for_all_nodes_with_given_os "$1"
               emit_stats "${1} nodes"
        elif [ "$2" != "all" ]; then
               get_counts_for_all_nodes_with_given_role "$2"
               emit_stats "${2} nodes"
       fi
}

print_all_core_stats() {
        get_and_emit_counts "all" "all"
        get_and_emit_counts "linux" "all"
        get_and_emit_counts "windows" "all"
        get_and_emit_counts "linux" "manager"
        get_and_emit_counts "linux" "worker"
        get_and_emit_counts "windows" "worker"
        get_and_emit_counts "all" "worker"
}

if [ `docker node inspect self |jq '.[].ManagerStatus.Leader'` == "true" ]; then
        echo "Category  NodeCount       CoreCount       AverageCoresPerNode     MaxCoresOnNode  MinCoresOnNode"
        print_all_core_stats
fi
